/*
 * Decompiled with CFR 0.152.
 */
package com.android.server.usage;

import android.app.usage.ConfigurationStats;
import android.app.usage.TimeSparseArray;
import android.app.usage.UsageEvents;
import android.app.usage.UsageStats;
import android.content.res.Configuration;
import android.os.SystemClock;
import android.util.ArraySet;
import android.util.Slog;
import com.android.server.usage.IntervalStats;
import com.android.server.usage.UnixCalendar;
import com.android.server.usage.UsageStatsDatabase;
import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

class UserUsageStatsService {
    private static final String TAG = "UsageStatsService";
    private static final boolean DEBUG = false;
    private static final SimpleDateFormat sDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    private final UsageStatsDatabase mDatabase;
    private final IntervalStats[] mCurrentStats;
    private boolean mStatsChanged = false;
    private final UnixCalendar mDailyExpiryDate = new UnixCalendar(0L);
    private final StatsUpdatedListener mListener;
    private final String mLogPrefix;
    private static final UsageStatsDatabase.StatCombiner<UsageStats> sUsageStatsCombiner = new UsageStatsDatabase.StatCombiner<UsageStats>(){

        @Override
        public void combine(IntervalStats stats, boolean mutable, List<UsageStats> accResult) {
            if (!mutable) {
                accResult.addAll(stats.packageStats.values());
                return;
            }
            int statCount = stats.packageStats.size();
            for (int i = 0; i < statCount; ++i) {
                accResult.add(new UsageStats(stats.packageStats.valueAt(i)));
            }
        }
    };
    private static final UsageStatsDatabase.StatCombiner<ConfigurationStats> sConfigStatsCombiner = new UsageStatsDatabase.StatCombiner<ConfigurationStats>(){

        @Override
        public void combine(IntervalStats stats, boolean mutable, List<ConfigurationStats> accResult) {
            if (!mutable) {
                accResult.addAll(stats.configurations.values());
                return;
            }
            int configCount = stats.configurations.size();
            for (int i = 0; i < configCount; ++i) {
                accResult.add(new ConfigurationStats(stats.configurations.valueAt(i)));
            }
        }
    };

    UserUsageStatsService(int userId, File usageStatsDir, StatsUpdatedListener listener) {
        this.mDatabase = new UsageStatsDatabase(usageStatsDir);
        this.mCurrentStats = new IntervalStats[4];
        this.mListener = listener;
        this.mLogPrefix = "User[" + Integer.toString(userId) + "] ";
    }

    void init(long currentTimeMillis) {
        this.mDatabase.init(currentTimeMillis);
        int nullCount = 0;
        for (int i = 0; i < this.mCurrentStats.length; ++i) {
            this.mCurrentStats[i] = this.mDatabase.getLatestUsageStats(i);
            if (this.mCurrentStats[i] != null) continue;
            ++nullCount;
        }
        if (nullCount > 0) {
            if (nullCount != this.mCurrentStats.length) {
                Slog.w(TAG, this.mLogPrefix + "Some stats have no latest available");
            }
            this.loadActiveStats(currentTimeMillis, false);
        } else {
            this.mDailyExpiryDate.setTimeInMillis(this.mCurrentStats[0].beginTime);
            this.mDailyExpiryDate.addDays(1);
            this.mDailyExpiryDate.truncateToDay();
            Slog.i(TAG, this.mLogPrefix + "Rollover scheduled @ " + sDateFormat.format(this.mDailyExpiryDate.getTimeInMillis()) + "(" + this.mDailyExpiryDate.getTimeInMillis() + ")");
        }
        for (IntervalStats stat : this.mCurrentStats) {
            int pkgCount = stat.packageStats.size();
            for (int i = 0; i < pkgCount; ++i) {
                UsageStats pkgStats = stat.packageStats.valueAt(i);
                if (pkgStats.mLastEvent != 1 && pkgStats.mLastEvent != 4) continue;
                stat.update(pkgStats.mPackageName, stat.lastTimeSaved, 3);
                this.notifyStatsChanged();
            }
            stat.updateConfigurationStats(null, stat.lastTimeSaved);
        }
    }

    void onTimeChanged(long oldTime, long newTime) {
        this.persistActiveStats();
        this.mDatabase.onTimeChanged(newTime - oldTime);
        this.loadActiveStats(newTime, true);
    }

    void reportEvent(UsageEvents.Event event) {
        if (event.mTimeStamp >= this.mDailyExpiryDate.getTimeInMillis()) {
            this.rolloverStats(event.mTimeStamp);
        }
        IntervalStats currentDailyStats = this.mCurrentStats[0];
        Configuration newFullConfig = event.mConfiguration;
        if (event.mEventType == 5 && currentDailyStats.activeConfiguration != null) {
            event.mConfiguration = Configuration.generateDelta(currentDailyStats.activeConfiguration, newFullConfig);
        }
        if (currentDailyStats.events == null) {
            currentDailyStats.events = new TimeSparseArray();
        }
        currentDailyStats.events.put(event.mTimeStamp, event);
        for (IntervalStats stats : this.mCurrentStats) {
            if (event.mEventType == 5) {
                stats.updateConfigurationStats(newFullConfig, event.mTimeStamp);
                continue;
            }
            stats.update(event.mPackage, event.mTimeStamp, event.mEventType);
        }
        this.notifyStatsChanged();
    }

    private <T> List<T> queryStats(int intervalType, long beginTime, long endTime, UsageStatsDatabase.StatCombiner<T> combiner) {
        if (intervalType == 4 && (intervalType = this.mDatabase.findBestFitBucket(beginTime, endTime)) < 0) {
            intervalType = 0;
        }
        if (intervalType < 0 || intervalType >= this.mCurrentStats.length) {
            return null;
        }
        IntervalStats currentStats = this.mCurrentStats[intervalType];
        if (beginTime >= currentStats.endTime) {
            return null;
        }
        long truncatedEndTime = Math.min(currentStats.beginTime, endTime);
        List<T> results = this.mDatabase.queryUsageStats(intervalType, beginTime, truncatedEndTime, combiner);
        if (beginTime < currentStats.endTime && endTime > currentStats.beginTime) {
            if (results == null) {
                results = new ArrayList<T>();
            }
            combiner.combine(currentStats, true, results);
        }
        return results;
    }

    List<UsageStats> queryUsageStats(int bucketType, long beginTime, long endTime) {
        return this.queryStats(bucketType, beginTime, endTime, sUsageStatsCombiner);
    }

    List<ConfigurationStats> queryConfigurationStats(int bucketType, long beginTime, long endTime) {
        return this.queryStats(bucketType, beginTime, endTime, sConfigStatsCombiner);
    }

    UsageEvents queryEvents(final long beginTime, final long endTime) {
        final ArraySet names = new ArraySet();
        List<UsageEvents.Event> results = this.queryStats(0, beginTime, endTime, new UsageStatsDatabase.StatCombiner<UsageEvents.Event>(){

            @Override
            public void combine(IntervalStats stats, boolean mutable, List<UsageEvents.Event> accumulatedResult) {
                if (stats.events == null) {
                    return;
                }
                int startIndex = stats.events.closestIndexOnOrAfter(beginTime);
                if (startIndex < 0) {
                    return;
                }
                int size = stats.events.size();
                for (int i = startIndex; i < size; ++i) {
                    if (stats.events.keyAt(i) >= endTime) {
                        return;
                    }
                    UsageEvents.Event event = (UsageEvents.Event)stats.events.valueAt(i);
                    names.add(event.mPackage);
                    if (event.mClass != null) {
                        names.add(event.mClass);
                    }
                    accumulatedResult.add(event);
                }
            }
        });
        if (results == null || results.isEmpty()) {
            return null;
        }
        Object[] table = names.toArray(new String[names.size()]);
        Arrays.sort(table);
        return new UsageEvents(results, (String[])table);
    }

    void persistActiveStats() {
        if (this.mStatsChanged) {
            Slog.i(TAG, this.mLogPrefix + "Flushing usage stats to disk");
            try {
                for (int i = 0; i < this.mCurrentStats.length; ++i) {
                    this.mDatabase.putUsageStats(i, this.mCurrentStats[i]);
                }
                this.mStatsChanged = false;
            }
            catch (IOException e) {
                Slog.e(TAG, this.mLogPrefix + "Failed to persist active stats", e);
            }
        }
    }

    private void rolloverStats(long currentTimeMillis) {
        long startTime = SystemClock.elapsedRealtime();
        Slog.i(TAG, this.mLogPrefix + "Rolling over usage stats");
        Configuration previousConfig = this.mCurrentStats[0].activeConfiguration;
        ArraySet<String> continuePreviousDay = new ArraySet<String>();
        for (IntervalStats stat : this.mCurrentStats) {
            int pkgCount = stat.packageStats.size();
            for (int i = 0; i < pkgCount; ++i) {
                UsageStats pkgStats = stat.packageStats.valueAt(i);
                if (pkgStats.mLastEvent != 1 && pkgStats.mLastEvent != 4) continue;
                continuePreviousDay.add(pkgStats.mPackageName);
                stat.update(pkgStats.mPackageName, this.mDailyExpiryDate.getTimeInMillis() - 1L, 3);
                this.notifyStatsChanged();
            }
            stat.updateConfigurationStats(null, this.mDailyExpiryDate.getTimeInMillis() - 1L);
        }
        this.persistActiveStats();
        this.mDatabase.prune(currentTimeMillis);
        this.loadActiveStats(currentTimeMillis, false);
        int continueCount = continuePreviousDay.size();
        for (int i = 0; i < continueCount; ++i) {
            String name = (String)continuePreviousDay.valueAt(i);
            long beginTime = this.mCurrentStats[0].beginTime;
            for (IntervalStats stat : this.mCurrentStats) {
                stat.update(name, beginTime, 4);
                stat.updateConfigurationStats(previousConfig, beginTime);
                this.notifyStatsChanged();
            }
        }
        this.persistActiveStats();
        long totalTime = SystemClock.elapsedRealtime() - startTime;
        Slog.i(TAG, this.mLogPrefix + "Rolling over usage stats complete. Took " + totalTime + " milliseconds");
    }

    private void notifyStatsChanged() {
        if (!this.mStatsChanged) {
            this.mStatsChanged = true;
            this.mListener.onStatsUpdated();
        }
    }

    private void loadActiveStats(long currentTimeMillis, boolean force) {
        UnixCalendar tempCal = this.mDailyExpiryDate;
        for (int intervalType = 0; intervalType < this.mCurrentStats.length; ++intervalType) {
            tempCal.setTimeInMillis(currentTimeMillis);
            UnixCalendar.truncateTo(tempCal, intervalType);
            if (!force && this.mCurrentStats[intervalType] != null && this.mCurrentStats[intervalType].beginTime == tempCal.getTimeInMillis()) continue;
            long lastBeginTime = this.mDatabase.getLatestUsageStatsBeginTime(intervalType);
            this.mCurrentStats[intervalType] = lastBeginTime >= tempCal.getTimeInMillis() ? this.mDatabase.getLatestUsageStats(intervalType) : null;
            if (this.mCurrentStats[intervalType] != null) continue;
            this.mCurrentStats[intervalType] = new IntervalStats();
            this.mCurrentStats[intervalType].beginTime = tempCal.getTimeInMillis();
            this.mCurrentStats[intervalType].endTime = currentTimeMillis;
        }
        this.mStatsChanged = false;
        this.mDailyExpiryDate.setTimeInMillis(currentTimeMillis);
        this.mDailyExpiryDate.addDays(1);
        this.mDailyExpiryDate.truncateToDay();
        Slog.i(TAG, this.mLogPrefix + "Rollover scheduled @ " + sDateFormat.format(this.mDailyExpiryDate.getTimeInMillis()) + "(" + tempCal.getTimeInMillis() + ")");
    }

    private static String eventToString(int eventType) {
        switch (eventType) {
            case 0: {
                return "NONE";
            }
            case 2: {
                return "MOVE_TO_BACKGROUND";
            }
            case 1: {
                return "MOVE_TO_FOREGROUND";
            }
            case 3: {
                return "END_OF_DAY";
            }
            case 4: {
                return "CONTINUE_PREVIOUS_DAY";
            }
            case 5: {
                return "CONFIGURATION_CHANGE";
            }
        }
        return "UNKNOWN";
    }

    static interface StatsUpdatedListener {
        public void onStatsUpdated();
    }
}

